<?php
session_start();
include 'includes/db_connect.php';

// Check CSRF Token
if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
    die("Invalid CSRF token.");
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $full_name = htmlspecialchars(strip_tags($_POST['full_name']));
    $email = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
    $phone = htmlspecialchars(strip_tags($_POST['phone']));
    $address = htmlspecialchars(strip_tags($_POST['address']));
    
    // Ticket Quantities
    $adult_qty = (int)$_POST['tickets']['adult'];
    $child_teen_qty = (int)$_POST['tickets']['child_teen'];
    $child_free_qty = (int)$_POST['tickets']['child_free'];
    
    $total_requested = $adult_qty + $child_teen_qty + $child_free_qty;

    if ($total_requested <= 0) {
        die("Please select at least one ticket.");
    }

    // CHECK AVAILABILITY
    // Get Limit
    $limitStmt = $pdo->query("SELECT setting_value FROM settings WHERE setting_key = 'total_tickets'");
    $limitRow = $limitStmt->fetch();
    $totalLimit = $limitRow ? (int)$limitRow['setting_value'] : 800;

    // Get Sold
    $soldStmt = $pdo->query("SELECT COUNT(*) FROM tickets WHERE status != 'cancelled'");
    $soldCount = $soldStmt->fetchColumn();

    if (($soldCount + $total_requested) > $totalLimit) {
        die("Sorry, there are only " . ($totalLimit - $soldCount) . " tickets remaining. You requested " . $total_requested . ".");
    }
    
    // Calculate Total (Backend Validation)
    $total_amount = ($adult_qty * 60) + ($child_teen_qty * 20) + ($child_free_qty * 0);
    
    // Generate Unique Booking Reference
    $booking_ref = 'CAL-' . strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 8));
    
    try {
        $pdo->beginTransaction();
        
        // Insert Booking
        $stmt = $pdo->prepare("INSERT INTO bookings (booking_ref, full_name, email, phone, address, total_amount, payment_status) VALUES (?, ?, ?, ?, ?, ?, 'pending')");
        $stmt->execute([$booking_ref, $full_name, $email, $phone, $address, $total_amount]);
        $booking_id = $pdo->lastInsertId();
        
        // Insert Booking Items
        $items = [
            ['type' => 'Adult', 'qty' => $adult_qty, 'price' => 60],
            ['type' => 'Teen (10-17)', 'qty' => $child_teen_qty, 'price' => 20],
            ['type' => 'Child (0-9)', 'qty' => $child_free_qty, 'price' => 0]
        ];
        
        $itemStmt = $pdo->prepare("INSERT INTO booking_items (booking_id, ticket_type, quantity, price) VALUES (?, ?, ?, ?)");
        
        foreach ($items as $item) {
            if ($item['qty'] > 0) {
                $itemStmt->execute([$booking_id, $item['type'], $item['qty'], $item['price']]);
            }
        }
        
        // If total is 0 (all free tickets), mark as paid immediately
        if ($total_amount == 0) {
            $stmt = $pdo->prepare("UPDATE bookings SET payment_status = 'paid' WHERE id = ?");
            $stmt->execute([$booking_id]);
            
            // Generate tickets immediately (Redirect to confirmation logic)
            // Ideally, calling a function or redirecting to a specific handler
            
            $pdo->commit();
            header("Location: ../../secure/verify/payment?ref=$booking_ref&internal_redirect=1");
            exit();
        } else {
            // Redirect to Mock Payment Gateway
            $pdo->commit();
            header("Location: ../../secure/gateway/pay?ref=$booking_ref&amount=$total_amount");
            exit();
        }
        
    } catch (Exception $e) {
        $pdo->rollBack();
        die("Error processing booking: " . $e->getMessage());
    }
}
?>