<?php
session_start();
include '../includes/db_connect.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: index.php");
    exit();
}

$message = '';
$messageType = '';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $code = trim($_POST['ticket_code']);
    
    // Auto-correct common typos (S->5, O->0) for manual entry
    $code = str_replace(['S', 'O'], ['5', '0'], strtoupper($code));
    
    if (!empty($code)) {
        // Check ticket
        $stmt = $pdo->prepare("SELECT t.*, b.full_name FROM tickets t JOIN bookings b ON t.booking_id = b.id WHERE t.ticket_code = ?");
        $stmt->execute([$code]);
        $ticket = $stmt->fetch();
        
        if ($ticket) {
            if ($ticket['status'] == 'valid') {
                // Mark used
                $update = $pdo->prepare("UPDATE tickets SET status = 'used', scanned_at = NOW() WHERE id = ?");
                $update->execute([$ticket['id']]);
                
                $message = "VALID TICKET! <br> Attendee: {$ticket['full_name']} <br> Type: {$ticket['ticket_type']}";
                $messageType = 'success';
            } elseif ($ticket['status'] == 'used') {
                $message = "WARNING: TICKET ALREADY USED! <br> Scanned at: {$ticket['scanned_at']}";
                $messageType = 'warning';
            } else {
                $message = "INVALID: Ticket Cancelled.";
                $messageType = 'error';
            }
        } else {
            $message = "ERROR: Ticket not found.";
            $messageType = 'error';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ticket Scanner - Calabar Day</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <script src="https://unpkg.com/html5-qrcode" type="text/javascript"></script> 
    <style>
        body { display: flex; align-items: center; justify-content: center; min-height: 100vh; background: #000; flex-direction: column; }
        .scanner-box { 
            background: #111; padding: 20px; border-radius: 20px; 
            border: 2px solid var(--secondary-color); 
            text-align: center; width: 100%; max-width: 500px;
            margin-top: 20px;
            position: relative;
        }
        .scan-input { 
            width: 100%; padding: 15px; font-size: 1.2rem; text-align: center; 
            margin-bottom: 20px; border-radius: 10px; border: none; 
            font-family: monospace; letter-spacing: 2px;
        }
        .result-box { 
            padding: 20px; border-radius: 10px; margin-top: 20px; font-weight: bold; font-size: 1.2rem;
        }
        .success { background: #4CAF50; color: white; }
        .warning { background: #FFC107; color: black; }
        .error { background: #F44336; color: white; }
        
        #reader { width: 100%; margin-bottom: 20px; border-radius: 10px; overflow: hidden; display: none; }
        
        .suggestions-list {
            position: absolute;
            width: calc(100% - 40px);
            background: #fff;
            color: #000;
            max-height: 150px;
            overflow-y: auto;
            border-radius: 0 0 10px 10px;
            z-index: 1000;
            top: 232px; /* Adjust based on input position */
            left: 20px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.3);
            display: none;
            text-align: left;
        }
        .suggestion-item {
            padding: 10px;
            cursor: pointer;
            border-bottom: 1px solid #eee;
            font-family: monospace;
        }
        .suggestion-item:hover {
            background-color: #f0f0f0;
        }
        .scan-btn-toggle {
            background: transparent; border: 1px solid var(--secondary-color); color: var(--secondary-color);
            margin-bottom: 15px; padding: 10px; border-radius: 5px; cursor: pointer; display: inline-block;
        }
    </style>
</head>
<body>

    <div class="scanner-box">
        <h2 style="color: var(--secondary-color); margin-bottom: 10px;">QR Validation</h2>
        <a href="panel" style="color: #ccc; text-decoration: underline; margin-bottom: 20px; display: block; font-size: 0.9rem;">Back to Dashboard</a>
        
        <button id="toggleScanner" class="scan-btn-toggle" onclick="toggleCamera()">
            <i class="fas fa-camera"></i> Show Camera Scanner
        </button>
        
        <div id="reader"></div>
        
        <form method="POST" id="scanForm" autocomplete="off">
            <div style="position: relative;">
                <input type="text" id="ticketInput" name="ticket_code" class="scan-input" placeholder="Scan or Type Last 5 Digits" autofocus onkeyup="searchTickets(this.value)">
                <div id="suggestions" class="suggestions-list"></div>
            </div>
            <button type="submit" class="btn-primary" style="width: 100%;">VALIDATE</button>
        </form>
        
        <?php if($message): ?>
            <div class="result-box <?php echo $messageType; ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Camera Scanner Logic
        let html5QrcodeScanner = null;
        
        function toggleCamera() {
            const reader = document.getElementById('reader');
            const btn = document.getElementById('toggleScanner');
            
            if (reader.style.display === 'none' || reader.style.display === '') {
                reader.style.display = 'block';
                btn.innerText = "Check Camera"; // Feedback
                
                // Init scanner
                if (!html5QrcodeScanner) {
                    html5QrcodeScanner = new Html5QrcodeScanner(
                        "reader", { fps: 10, qrbox: 250 });
                    
                    html5QrcodeScanner.render(onScanSuccess, onScanFailure);
                }
            } else {
                // Just toggling visibility for now, proper stop/start can be added if needed
                // reader.style.display = 'none'; // Keeping it simple
            }
        }
        
        function onScanSuccess(decodedText, decodedResult) {
            // Handle on success condition with the decoded text or result.
            document.getElementById('ticketInput').value = decodedText;
            document.getElementById('scanForm').submit();
            
            // Should stop scanning?
            // html5QrcodeScanner.clear();
        }

        function onScanFailure(error) {
            // handle scan failure, usually better to ignore and keep scanning.
            // console.warn(`Code scan error = ${error}`);
        }
        
        // Autocomplete Logic
        function searchTickets(query) {
            const suggestionsBox = document.getElementById('suggestions');
            
            if(query.length < 3) {
                suggestionsBox.style.display = 'none';
                return;
            }
            
            // Only search if it looks like they are typing manually (not a full scan paste)
            if (query.length > 15) { 
                suggestionsBox.style.display = 'none';
                return; 
            }

            fetch('search?q=' + query)
                .then(response => response.json())
                .then(data => {
                    suggestionsBox.innerHTML = '';
                    if (data.length > 0) {
                        suggestionsBox.style.display = 'block';
                        data.forEach(ticket => {
                            const div = document.createElement('div');
                            div.className = 'suggestion-item';
                            // Highlight match if possible, for now just show code
                            div.innerText = ticket.ticket_code + ' (' + ticket.status + ')';
                            div.onclick = function() {
                                document.getElementById('ticketInput').value = ticket.ticket_code;
                                suggestionsBox.style.display = 'none';
                                // Optional: Auto submit? Let's let them click Validate to be safe
                                // document.getElementById('scanForm').submit();
                            };
                            suggestionsBox.appendChild(div);
                        });
                    } else {
                         suggestionsBox.style.display = 'none';
                    }
                })
                .catch(error => console.error('Error:', error));
        }

        // Hide suggestions when clicking outside
        document.addEventListener('click', function(e) {
            if (e.target.id !== 'ticketInput') {
                document.getElementById('suggestions').style.display = 'none';
            }
        });
    </script>

</body>
</html>
